//
//  Library for the universal signal machine simulator.
//
// This is the main function to generate a universal signal machine for a given set of speeds.
// The function is so big, it has its own separate file for clarity.
//

.create_universal_simulator_speed .= (
	speed_list  // list of speeds
) -> {

	:=  // return 
	create_signal_machine {  //create a signal machine

	.speed_list .= list_sort_unique ( speed_list ) ;
	.SPEED_NBR .= # speed_list ;
	// This is a list of all the indices for speeds.
	// It is only generated for optimization purposes.
	.speed_indices_list := [] ;
	{
		.i := 0 ;
		foreach ( tmp : speed_list ) {
			speed_indices_list := speed_indices_list _ i ;
			i := i + 1 ;
		} ;
	} ;
	.speed_indices_list .= speed_indices_list ; // make it a constant
	.OS .= list_sorted_unique_list_all_sublist ( speed_indices_list ) ;
		
	.SPEED_MIN .= list_min ( speed_list ) ;
	.SPEED_MAX .= list_max ( speed_list ) ;
	.SPEED_DIFF_MIN .= list_min_diff ( speed_list ) ;
	.SPEED_ABS_MAX .= max ( abs ( SPEED_MIN ) , abs ( SPEED_MAX ) ) ;
	
	.SPEED_RAPID .= 40 * SPEED_ABS_MAX ;
	.SPEED_RAPID_NEG .= - SPEED_RAPID ;
	.SPEED_RAPID_HALF .= SPEED_RAPID / 2 ;
	.SPEED_RAPID_NEG_HALF .= SPEED_RAPID_NEG / 2 ;
	
	.COEF_SPEED_CHECK .= 99 / 100 ; // Maximal % of height between middle point and the encounter point relatively to start (origin here) and encounter
	
	.SPEED_CHECK_UP := SPEED_RAPID_HALF ;  // There should be a value in case there are only 2 speeds
	foreach ( i : speed_indices_list ) {
		foreach ( k : 1 ... i - 1 ) {
			.height_middle .= ( speed_list[ i ] - speed_list [ k - 1 ] ) / ( SPEED_RAPID - speed_list [ k - 1 ] ) * COEF_SPEED_CHECK ;
			SPEED_CHECK_UP := max ( SPEED_CHECK_UP ,
				( speed_list[ k ] * ( height_middle - 1 ) + speed_list [ i ] ) / height_middle ) ;
		} ;
	} ;
	.SPEED_CHECK_UP .=  SPEED_CHECK_UP ;  // to make it constant
	
	//---------------------------------------
	//
	//   META-SIGNALS
	//
	//---------------------------------------
		
	//---------------------------------------
	//   ENCODING
	//---------------------------------------
	
	foreach ( i : speed_indices_list ) {
		add_meta_signal ( ms_border_left ( i ) , speed_list [ i ] ) { 
			color => COLOR_BORDER ; } ;
		add_meta_signal ( ms_id ( i ) , speed_list [ i ] ) { 
			color => COLOR_ID ; } ;
		foreach ( os : OS ) {
			add_meta_signal ( ms_main_set ( i , os ) , speed_list [ i ] ) { 
				color => COLOR_MAIN ;	} ;
		} ;
		add_meta_signal ( ms_border_right ( i ) , speed_list [ i ] ) { 
			color => COLOR_BORDER ; } ;
		//
		//  RULES
		//
		add_meta_signal ( ms_rule_middle ( i ) , speed_list [ i ] ) { 
			color => COLOR_RULE_MIDDLE ;
		} ;
		add_meta_signal ( ms_rule_bound ( i ) , speed_list [ i ] ) { 
			color => COLOR_RULE_BOUND ; } ;
		foreach ( l : speed_indices_list ) {
			add_meta_signal ( ms_if ( i , l  ) , speed_list [ i ] ) { 
				color => COLOR_IF ;
			} ;
			add_meta_signal ( ms_then ( i , l ) , speed_list [ i ] ) { 
				color => COLOR_THEN_ID ; } ;
		} ;
	} ;
	

	//---------------------------------
	//   SHRINKING
	//---------------------------------
	
	.SPEED_SHRINK_TOP .= 20 * SPEED_ABS_MAX ; 
	.SPEED_SHRINK_BOT .= 3 * SPEED_SHRINK_TOP ;
	.SPEED_SHRINK_BACK .= - 3 * SPEED_SHRINK_TOP ;
	
	foreach ( i : speed_indices_list ) {
		foreach ( r : RIGHTLEFT ) {
			.RPOS := 0  ; // create variable
			if ( r == RIGHT ) {
				RPOS := -1 ;
			} else {
				RPOS := 1 ;
			} ;
			add_meta_signal ( ms_shrink_bottom ( r , i ) , RPOS * SPEED_SHRINK_BOT + speed_list [ i ] ) {  
				color => Black ; 
				line_style => "densely dotted" ; } ;
			add_meta_signal ( ms_shrink_bottom_both ( r , i ) , RPOS * SPEED_SHRINK_BOT + speed_list [ i ] ) { 
				color => Black ;
				line_style => "densely dotted" ;  } ;
			add_meta_signal ( ms_shrink_top ( r , i ) , RPOS * SPEED_SHRINK_TOP + speed_list [ i ] ) { 
				color => COLOR_BORDER ; } ;
			add_meta_signal ( ms_shrink_delayed ( r , i ) , ( RPOS * SPEED_SHRINK_TOP + speed_list [ i ] ) / 2 ) { 
				color => COLOR_BORDER ; } ;
			add_meta_signal ( ms_shrink_top_test ( r , i ) , RPOS * SPEED_SHRINK_TOP + speed_list [ i ] ) { 
				color => COLOR_BORDER ; } ;
			add_meta_signal ( ms_shrink_back ( r , i ) , RPOS * SPEED_SHRINK_BACK + speed_list [ i ] ) { 
				color => Black ;
				line_style => "densely dotted" ;  } ;
		} ;
		add_meta_signal ( ms_shrink_id ( i ) , SPEED_SHRINK_TOP + speed_list [ i ] ) { 
			color => COLOR_ID ; }  ;
		add_meta_signal ( ms_shrink_rule_bound ( i ) , - SPEED_SHRINK_TOP + speed_list [ i ] ) { 
			color => COLOR_RULE_BOUND ; } ;
		add_meta_signal ( ms_shrink_rule_middle ( i ) , - SPEED_SHRINK_TOP + speed_list [ i ] ) { 
			color => COLOR_RULE_MIDDLE_OK ; } ;
		foreach ( l : speed_indices_list ) {
			add_meta_signal ( ms_shrink_if ( i , l ) , - SPEED_SHRINK_TOP + speed_list [ i ] ) { 
				color => COLOR_IF_ID_NOT_MET ; } ;
			add_meta_signal ( ms_shrink_then ( i , l ) , - SPEED_SHRINK_TOP + speed_list [ i ] ) { 
				color => COLOR_THEN_ID ; } ;
			add_meta_signal ( ms_shrink_test ( i , l ) , ( speed_list [ i ] + speed_list [ l ] ) /2 ) {  
				color => COLOR_SHRINK_TEST ;	} ;
		} ;
		add_meta_signal ( ms_shrink_test_ok ( i ) , speed_list [ i ]  ) { 
			color => COLOR_SHRINK_TEST ;	} ;
		add_meta_signal ( ms_shrink_test_fail ( i ) , speed_list [ i ] ) { 
			line_style => "dashed" ; 
			color => Red ;	} ;
		add_meta_signal ( ms_shrink_test_left( i ) , SPEED_SHRINK_BACK + speed_list [ i ] ) { 
			color => COLOR_SHRINK_TEST ; } ;
		add_meta_signal ( ms_shrink_test_right ( i ) , - SPEED_SHRINK_BACK + speed_list [ i ] ) {
			color => COLOR_SHRINK_TEST ; } ;
		add_meta_signal ( ms_shrink_order ( i ) , SPEED_SHRINK_TOP + speed_list [ i ] ) { 
			color => Red ; } ;
	} ;


	//---------------------------------
	//   TEST
	//---------------------------------
		
	foreach ( i : speed_indices_list ) {
		add_meta_signal ( ms_test_start ( i ) , - SPEED_SHRINK_TOP + speed_list [ i ] ) { 
			color => COLOR_CHECK ; } ;
		.SPEED_TEST_UP .= ( speed_list[i] + SPEED_ABS_MAX * ( 1 + 9/10 * TEST_EPSILON ) ) / ( 9/10 * TEST_EPSILON ) ;
		.delta := SPEED_ABS_MAX ;
		foreach ( j : 0 ... SPEED_NBR - 2 ) {
			.tt .= speed_list [ j + 1 ]  - speed_list [ j ] ;
			if ( delta < tt ) {
				delta := tt ;
			} ;
		} ;
		.T_T .= 2 * SPEED_ABS_MAX / SPEED_RAPID + 4 * SPEED_ABS_MAX / delta ;
		.T_L .= -1 + TEST_EPSILON ;
		.T_R .= T_L ;
		.X_L .= - ( T_T - T_L ) * SPEED_ABS_MAX ;
		.X_R .= ( T_T - T_R ) * SPEED_ABS_MAX ;
		.SPEED_TEST_RIGHT_UP .= ( X_R + speed_list[i] ) / TEST_EPSILON ;
		.SPEED_TEST_LEFT_BACK .= ( 3/2 * TEST_EPSILON * speed_list[i] - speed_list[i] - X_L ) / ( TEST_EPSILON / 2 ) ;
		.SPEED_TEST_RIGHT_BACK .= ( 2 * TEST_EPSILON * speed_list[i] - speed_list[i] - X_R ) / TEST_EPSILON ;
		add_meta_signal ( ms_test_left ( i ) , ( X_L + speed_list[i] ) / TEST_EPSILON ) { 
			color => COLOR_TEST_LEFT ;
		} ;
		add_meta_signal ( ms_test_left_ok ( i ) , SPEED_TEST_LEFT_BACK ) { 
			color =>  COLOR_TEST_LEFT_OK ; 
		} ;
		add_meta_signal ( ms_test_left_fail ( i ) , SPEED_TEST_LEFT_BACK ) { 
			line_style => "dashed" ; 
			color =>  COLOR_TEST_LEFT_FAIL ; } ;
		add_meta_signal ( ms_test_right_fail ( i ) , SPEED_TEST_RIGHT_BACK ) { 
			line_style => "densely dashed" ;
			color =>  COLOR_TEST_RIGHT_FAIL ; 
		} ;
		add_meta_signal ( ms_test_right_up ( i ) , SPEED_TEST_RIGHT_UP ) { 
			color => COLOR_TEST_RIGHT_UP ; } ;
		add_meta_signal ( ms_test_right ( i , i , i ) , 2 * SPEED_TEST_RIGHT_UP ) { 
			color =>  COLOR_TEST_RIGHT; } ;
		foreach ( k : 0 ... i ) {
			add_meta_signal ( ms_test_right_ok ( i , k ) , SPEED_TEST_RIGHT_BACK ) { 
				color => COLOR_TEST_RIGHT_OK ; } ;
		} ;
		foreach ( k : 0 ... i - 1 ) {
			.T_U .= -( 2 * SPEED_TEST_RIGHT_UP - speed_list [ i ] ) / ( 2 * SPEED_TEST_RIGHT_UP - speed_list [ k ] ) ;
			.X_U .= speed_list [ k ] * T_U ;
			.SPEED_RIGHT .= ( X_R - X_U ) / ( T_R - T_U ) ;
			foreach ( j : 0 ... k ) {
				add_meta_signal ( ms_test_right ( i , k , j ) , SPEED_RIGHT ) { 
					color =>  COLOR_TEST_RIGHT ; } ;
				add_meta_signal ( ms_test_right_wait ( i , k , j ) , SPEED_RIGHT ) { 
					color =>  COLOR_TEST_RIGHT ; } ;
			} ;
			add_meta_signal ( ms_test_left_up ( i , k ) , ( X_L - X_U ) / ( T_L - T_U ) ) { 
				color => COLOR_TEST_LEFT_UP ; } ;
			if ( 0 == k ) {
				add_meta_signal ( ms_test_left_up_fail ( i ) , ( X_L - X_U ) / ( TEST_EPSILON - T_U ) ) {
					color => COLOR_TEST_LEFT_FAIL ; } ;
			} ;
		} ;
		add_meta_signal ( ms_main_test_fail_r ( i ) , speed_list [ i ] ) { 
			line_style => "densely dashed" ;
			color => COLOR_MAIN_TEST_FAIL ; } ;
		add_meta_signal ( ms_main_test_fail_l ( i ) , speed_list [ i ] ) { 
			line_style => "densely dashed" ;
			color => COLOR_MAIN_TEST_FAIL ; } ;
		add_meta_signal ( ms_test_main_ok ( i ) , speed_list [ i ] ) { 
			color => COLOR_MAIN_TEST_OK ; } ;
	} ;
	

	//---------------------------------------
	//   CHECKING
	//---------------------------------------
					
	foreach ( i : speed_indices_list ) {
		foreach ( j : 0 ... i - 1 ) {
			// j -> rightmost
			// i -> leftmost
			add_meta_signal ( ms_check_maybe ( i , j ) , SPEED_RAPID ) { 
				color => COLOR_CHECK ; } ;
			add_meta_signal ( ms_check ( i , j ) , SPEED_RAPID ) { 
				color => COLOR_CHECK ; } ;
			add_meta_signal ( ms_check_up ( i , j ) , SPEED_CHECK_UP ) { 
				color => COLOR_CHECK ;} ;
			add_meta_signal ( ms_check_ok ( i , j ) , - SPEED_RAPID ) { 
				color => COLOR_CHECK ; } ;
			add_meta_signal ( ms_check_fail ( i , j ) , - SPEED_RAPID ) { 
				line_style => "densely dashed" ;
				color =>  COLOR_TEST_RIGHT_FAIL ; } ;
			// Coordinates of intersect ms_check with main _s2
			// with init of ms_check_up at (0,0) 
			// and junction of main at t = 1
			.t0 .= ( speed_list [ i ] - speed_list [ j ] ) / ( SPEED_RAPID - speed_list [ j ] ) ;
			.x0 .= SPEED_RAPID * t0;
			foreach ( k : j+1 ... i-1 ) {
				// Coordinates of intersect ms_check_up with main _s3
				// with init of ms_check_up at (0,0) 
				// and junction of main at t = 1
				.t1 .= ( speed_list [ i ] - speed_list [ k ] ) / ( SPEED_CHECK_UP - speed_list [ k ] ) ;
				.x1 .= SPEED_CHECK_UP * t1 ;
				add_meta_signal ( ms_check_intersect ( i, k, j ) ,  
					( x1 - x0 ) / ( t1 - t0 ) 
				) {
					color => COLOR_CHECK ; } ;
			} ;
		} ;
	} ;
	
	
	//---------------------------------------
	//   DISPOSAL AND APPLACTION OF ID'S
	//---------------------------------------
	
	foreach ( i : speed_indices_list ) {
		foreach ( j : 0 ... i - 1 ) {
			// j -> rightmost
			// i -> leftmost
			add_meta_signal ( ms_collect ( i , j  ) , SPEED_RAPID ) { 
				color => COLOR_COLLECT ; } ;
		} ;
		add_meta_signal ( ms_cross ( i ) , SPEED_RAPID_NEG ) { 
			color => COLOR_CROSS ;	} ;
		add_meta_signal ( ms_cross_ok ( i ) , SPEED_RAPID_NEG ) { 
			color => COLOR_CROSS ;	
			line_style => "densely dotted" ; } ;
		add_meta_signal ( ms_cross_back_ok ( i  ) , SPEED_RAPID ) { 
			color => COLOR_CROSS ;
			line_style => "densely dotted" ;	} ;
		add_meta_signal ( ms_cross_back ( i ) , SPEED_RAPID ) { 
			color => COLOR_CROSS ; 	} ;
		add_meta_signal ( ms_rule_bound_fail ( i ) , speed_list [ i ] ) { 
			line_style => "densely dashed" ;
			color => COLOR_RULE_BOUND_FAIL ; } ;
		add_meta_signal ( ms_rule_middle_fail ( i ) , speed_list [ i ] ) { 
			color => COLOR_RULE_MIDDLE_FAIL ;
			line_style => "densely dashed" ;
		} ;
		foreach ( l : speed_indices_list ) {
			add_meta_signal ( ms_if_ok ( i , l ) , speed_list [ i ] ) { 
				color => COLOR_IF_OK ;
				line_style => "densely dotted" ;
			} ; 
		} ;
	};

	
	//---------------------------------------
	//   RULE SELECTION
	//---------------------------------------
	
	foreach ( i : speed_indices_list ) {
		foreach ( os : OS ) {
			add_meta_signal ( ms_ready ( i , os ) , SPEED_RAPID_NEG ) { 
				color => COLOR_READY ;
			} ;
			add_meta_signal ( ms_ready_no ( i , os ) , SPEED_RAPID_NEG ) { 
				color => COLOR_READY_NO ;
				line_style => "dashed" ; 
			};
		} ;	
		foreach ( l : speed_indices_list ) {
			add_meta_signal ( ms_id_selected ( i , l )  , speed_list [ i ] ) { 
				color => COLOR_ID_SELECTED_RIGHT ; } ;
			add_meta_signal ( ms_id_copy ( i , l ) , SPEED_RAPID_NEG_HALF ) { 
				color => COLOR_ID_SELECTED_LEFT ; } ;
		} ;
	} ;
	
	add_meta_signal ( ms_ready_up_1 (), SPEED_RAPID_NEG_HALF ) { 
		color => COLOR_READY_UP_1 ; } ;
	add_meta_signal ( ms_ready_up_2 (), SPEED_RAPID_NEG ) { 
		color => COLOR_READY_UP_2 ; } ;
		
		
	//---------------------------------------
	//   SETTING OUTPUT
	//---------------------------------------
					
	foreach ( os : OS ) {
		add_meta_signal ( ms_fast_right ( os ) , SPEED_RAPID - SPEED_MAX ) { 
			line_style => "densely dotted" ;
			color => COLOR_FAST ; } ;
		add_meta_signal ( ms_fast_left ( os ) , SPEED_RAPID_NEG - SPEED_MAX ) { 
			line_style => "densely dotted" ;
			color => COLOR_FAST ; } ;
	} ;


	
	//---------------------------------
	//
	// COLLISION RULES
	//
	//---------------------------------

	.add_rule .= ( A , B ) -> {
		A --> B ;
	} ;

	//---------------------------------
	//   SHRINKING
	//---------------------------------
	
	foreach ( i : speed_indices_list ) {
		foreach ( k : 0 ... i -1 ) {
			add_rule ( [ ms_border_right ( i ) , ms_border_left ( k ) ] , 
				[ 	ms_shrink_bottom_both ( LEFT , k  ) ,  ms_shrink_top_test ( LEFT , k ) ,
					ms_shrink_bottom_both ( RIGHT , i ) ,  ms_shrink_top_test ( RIGHT , i ) ,
					ms_shrink_test ( i , k )
				] ) ;
			add_rule ( [ ms_shrink_test ( i , k ) , ms_shrink_test_left ( k ) ] , 
				[ ms_shrink_test_ok ( k ) ] ) ;
			add_rule ( [ ms_shrink_test_ok ( k ) , ms_shrink_test_right ( i ) ] , 
				[ ms_test_start ( i ) ] ) ; 
			add_rule ( [ ms_shrink_test ( i , k ) , ms_shrink_test_right ( i ) , ms_shrink_test_left ( k )  ] , 
				[ ms_test_start ( i ) ] ) ; 
			add_rule ( [ ms_shrink_test ( i , k ) , ms_shrink_test_right ( i ) ] , 
				[ ms_shrink_test_fail ( i ) ] ) ;
			add_rule ( [ ms_shrink_test_fail ( i ) , ms_shrink_test_left ( k ) ] , 
				[ ms_shrink_order ( k ) ] ) ; // CANCEL
		} ;
		add_rule ( [ ms_shrink_bottom ( LEFT , i ) , ms_id ( i ) ] , 
			[ ms_shrink_bottom ( LEFT , i ) , ms_shrink_id ( i ) ] ) ;
		add_rule ( [ ms_shrink_bottom ( LEFT , i ) , ms_id ( i ) ] , 
			[ ms_shrink_bottom ( LEFT , i ) , ms_shrink_id ( i ) ] ) ;
		add_rule ( [ ms_shrink_bottom_both ( LEFT , i ) , ms_id ( i ) ] , 
			[ ms_shrink_bottom_both ( LEFT , i ) , ms_shrink_id ( i ) ] ) ;
		add_rule ( [ ms_shrink_back ( LEFT , i ) , ms_shrink_id ( i ) ] , 
			[ ms_shrink_back ( LEFT , i ) , ms_id ( i ) ] ) ;

		.switch_on_RIGHT_tab .= ( regular_tab , bent_tab ) -> {
			add_rule ( [ ms_shrink_bottom ( RIGHT , i ) ] _ regular_tab , 
				[ ms_shrink_bottom ( RIGHT , i ) ] _ bent_tab  ) ;
			add_rule ( [ ms_shrink_bottom_both ( RIGHT , i ) ] _ regular_tab  ,
				[ ms_shrink_bottom_both ( RIGHT , i ) ] _ bent_tab ) ;
			add_rule ( [ ms_shrink_back ( RIGHT , i ) ] _ bent_tab  ,
				[ ms_shrink_back ( RIGHT , i ) ] _ regular_tab ) ;
		} ;
		.switch_on_RIGHT .= ( regular , bent ) -> {
			switch_on_RIGHT_tab ( [ regular ] , [ bent ] ) ;
			foreach ( i : speed_indices_list ) {
				foreach ( k : 0 ... i - 1 ) {  // To deal with possible superpositions
					foreach ( j : 0 ... k - 1 ) {  // To deal with possible superpositions
						switch_on_RIGHT_tab ( [ regular , ms_test_right ( i, k , j ) ] , [ bent , ms_test_right ( i, k , j ) ] ) ;
						switch_on_RIGHT_tab ( [ regular , ms_test_right_up ( i ) ] , [ bent , ms_test_right ( i, k , j ) ] ) ;
					} ;
				} ;
			} ;
		};
		switch_on_RIGHT ( ms_rule_bound ( i ) , ms_shrink_rule_bound ( i ) ) ; 
		switch_on_RIGHT ( ms_rule_middle ( i ) , ms_shrink_rule_middle ( i ) ) ; 
		foreach ( l : speed_indices_list ) {
			switch_on_RIGHT ( ms_if ( i , l ) , ms_shrink_if ( i , l ) ) ;  
			switch_on_RIGHT ( ms_then ( i , l ) , ms_shrink_then ( i , l ) ) ; 
		} ;
		add_rule ( [ ms_shrink_bottom ( LEFT , i ) , ms_main ( i ) , ms_shrink_bottom ( RIGHT , i ) ] , 
			[ ms_shrink_back ( LEFT , i ) , ms_main ( i ) , ms_shrink_back ( RIGHT , i ) ] ) ;
		add_rule ( [ ms_shrink_bottom_both ( LEFT , i ) , ms_shrink_bottom_both ( RIGHT , i) , ms_main ( i ) ] , 
			[ ms_main ( i ) , ms_shrink_back ( LEFT , i ), ms_shrink_back ( RIGHT , i ) ] ) ;
		foreach ( r : RIGHTLEFT ) {
			add_rule ( [ ms_shrink_bottom ( r , i ) , ms_main ( i ) ] ,  
				[ ms_main ( i ) , ms_shrink_back ( r , i ) ] ) ;
			add_rule ( [ ms_shrink_bottom ( r , i ) , ms_main_test_fail_l ( i ) ] ,  
				[ ms_main_test_fail_l ( i ) , ms_shrink_back ( r , i ) ] ) ;
			add_rule ( [ ms_shrink_bottom ( r , i ) , ms_main_test_fail_r ( i ) ] ,  
				[ ms_main_test_fail_r ( i ) , ms_shrink_back ( r , i ) ] ) ;
			add_rule ( [ ms_shrink_bottom_both ( r , i ) , ms_main ( i ) ] ,  
				[ ms_shrink_bottom ( r , i ) , ms_main ( i ) , ms_shrink_back ( r , i ) ] ) ;
			add_rule ( [ ms_shrink_bottom_both ( r , i ) , ms_main_test_fail_l ( i ) ] , 
				[ ms_shrink_bottom ( r , i ) , ms_main_test_fail_l ( i ) , ms_shrink_back ( r , i ) ] ) ;
			add_rule ( [ ms_shrink_bottom_both ( r , i ) , ms_main_test_fail_r ( i ) ] , 
				[ ms_shrink_bottom ( r , i ) , ms_main_test_fail_r ( i ) , ms_shrink_back ( r , i ) ] ) ;
			//To declare variables exactly in this context
			. not_r := undef ;
			. boundary := undef  ;
			. other_boundary := undef  ;
			. ms_shrink_test_dir := undef  ;
			if ( r == RIGHT ) {
				not_r := LEFT ;
				boundary := ms_border_right ;
				other_boundary := ms_border_left ;
				ms_shrink_test_dir := ms_shrink_test_right ;
			} else {
				not_r := RIGHT ;
				boundary := ms_border_left ;
				other_boundary := ms_border_right ;
				ms_shrink_test_dir := ms_shrink_test_left ;
			} ;
			add_rule ( [ ms_shrink_bottom ( r , i ) , other_boundary ( i ) ] , 
				[ ms_shrink_bottom ( not_r , i ) , ms_shrink_top ( not_r , i ) ] ) ;
			add_rule ( [ ms_shrink_top_test ( r , i ) , ms_shrink_back ( r , i ) ] , 
				[ boundary ( i ) , ms_shrink_test_dir ( i ) ] ) ;
			add_rule ( [ ms_shrink_top ( r , i ) , ms_shrink_back ( r , i ) ] , 
				[ boundary ( i ) ] ) ;
			add_rule ( [ ms_shrink_bottom ( r , i ) , ms_shrink_bottom ( not_r , i ) ] , 
				[ ms_shrink_bottom ( r , i ) ] ) ;
			add_rule ( [ ms_shrink_bottom_both ( r , i ) , ms_shrink_bottom ( not_r , i) ] , 
				[ ms_shrink_bottom ( r , i  ) ] ) ;
		} ;
		add_rule ( [ ms_shrink_order ( i ) , ms_border_left ( i ) ] , 
			[ ms_shrink_bottom_both ( LEFT , i ) ,  ms_shrink_top ( LEFT , i ) ] ) ;
	} ;
	

	//---------------------------------
	//   TESTING
	//---------------------------------

	foreach ( i : 1 ... SPEED_NBR - 1 ) {
		add_rule ( [ ms_test_left_ok ( i ) , ms_main ( i ) ] , 
			[ ms_test_main_ok ( i ) ] ) ;
		foreach ( k : 0 ... i -1 ) {
			add_rule ( [ ms_test_right ( i , i , i ) , ms_main ( k ) ] , 
				[ ms_test_left_up( i , k ) , ms_main ( k ) , ms_test_right ( i , k , k ) ] ) ;
			add_rule ( [ ms_test_left ( i ) , ms_test_left_up ( i , k ) ] , 
				[ ms_test_left_ok ( i ) ] ) ;
			add_rule ( [ ms_test_right ( i , k , k ) , ms_test_right_up ( i ) ] , 
				[ ms_test_right_ok ( i , k ) ] ) ;
			foreach ( j : k + 1 ... i - 1 ) {
				add_rule ( [ ms_test_right ( i , j , k ) , ms_test_right_up ( i ) ] , 
					[ ms_test_right_ok ( i , k ) ] ) ;
				add_rule ( [ ms_test_right_up ( i ) , ms_test_right_wait ( i , j , k ) , ms_main ( k ) ] , 
					[ ms_test_right_ok ( i , k ) , ms_main ( k ) ] ) ;
			} ; 
			add_rule ( [ ms_test_right_ok ( i , k ) ,  ms_test_main_ok ( i ) ] ,  
				[ ms_main ( i ) , ms_check_maybe ( i , k ) , ms_check_up ( i , k ) ] ) ;
		} ;	
		add_rule ( [ ms_test_start ( i ) , ms_main ( i ) ] , 
			[ ms_main ( i ) , ms_test_left ( i ) , ms_test_right_up ( i ) , ms_test_right ( i , i , i ) ] ) ;
		add_rule ( [ ms_test_start ( i ) , ms_main ( i ) , ms_shrink_bottom ( RIGHT , i ) ] , // Superposition
			[ ms_main ( i ) , ms_test_left ( i ) , ms_test_right_up ( i ) , ms_test_right ( i , i , i ) , ms_shrink_back ( RIGHT , i ) ] ) ;
		foreach ( j : 0 ... i -1 ) {
			add_rule ( [ ms_test_right ( i , i , i ) , ms_border_left ( j ) ] , 
				[ ms_test_right ( i , i , i ) , ms_shrink_bottom ( LEFT , j ) , ms_shrink_top ( LEFT , j ) ] ) ;
			add_rule ( [ ms_test_right ( i , i , i ) , ms_shrink_top ( RIGHT , j ) ] , 
				[ ms_test_right ( i , i , i ) , ms_shrink_top ( RIGHT , j ) , ms_shrink_delayed ( RIGHT , j ) ] ) ;
			add_rule ( [ ms_test_right ( i , i , i ) , ms_shrink_top ( LEFT , j ) ] , 
				[ ms_test_right ( i , i , i ) , ms_shrink_top ( LEFT , j ) , ms_shrink_delayed ( LEFT , j ) ] ) ;
			add_rule ( [ ms_test_right ( i , j , j ) , ms_border_right ( j ) ] , 
				[ ms_test_right ( i , j , j ) , ms_shrink_bottom ( RIGHT , j ) , ms_shrink_top ( RIGHT , j ) ] ) ;
			add_rule ( [ ms_test_right ( i , j , j ) , ms_shrink_top ( RIGHT , j ) ] , 
				[ ms_test_right ( i , j , j ) , ms_shrink_top ( RIGHT , j ) , ms_shrink_delayed ( RIGHT , j ) ] ) ;
			add_rule ( [ ms_test_right ( i , j , j ) , ms_shrink_top ( LEFT , j ) ] , 
				[ ms_test_right ( i , j , j ) , ms_shrink_top ( LEFT , j ) , ms_shrink_delayed ( LEFT , j ) ] ) ;
			add_rule ( [ ms_border_right ( j ) , ms_shrink_delayed ( RIGHT , j ) ] , 
				[ ms_shrink_bottom ( RIGHT , j ) , ms_shrink_top ( RIGHT , j )  ] ) ;
			add_rule ( [ ms_border_left ( j ) , ms_shrink_delayed ( LEFT , j ) ] , 
				[ ms_shrink_bottom ( LEFT , j ) , ms_shrink_top ( LEFT , j )  ] ) ;
			foreach ( k : j ... i - 1 ) {
				// Shrink on the right
				add_rule ( [ ms_test_right_wait ( i , k , j ) , ms_main( j ) ] , 
					[ ms_test_right ( i , k , j ) , ms_main( j ) ] ) ;
				foreach ( l : 0 ... k - 1 ) {
					add_rule ( [ ms_test_right ( i , k , j ) , ms_border_left ( l ) ] , 
						[ ms_test_right_wait ( i , k , l ) , ms_shrink_bottom ( LEFT , l ) , ms_shrink_top ( LEFT , l ) ] ) ;
				} ;
			} ;
		} ;
		//
		// FAIL on LEFT
		//
		foreach ( k : 0 ... i -1 ) {
			add_rule ( [ ms_test_left ( i ) , ms_test_left_up ( i , k ) ] , 
				[ ms_test_left_ok ( i ) ] ) ;
			add_rule ( [ ms_test_left_fail ( i ) , ms_test_left_up ( i , k ) ] , 
				[ ms_test_left_fail ( i ) ] ) ;
			add_rule ( [ ms_main_test_fail_l ( i ) , ms_test_left_up ( i , k ) ] , 
				[ ms_main_test_fail_l ( i ) ] ) ;
			.j := 0 ;
			add_rule ( [ ms_main_test_fail_l ( i ) , ms_test_right_ok ( i , k ) ] ,  
				[ ms_main ( i ) ] ) ;
			add_rule ( [ ms_test_left_fail ( i ) , ms_main ( i ) , ms_test_right_ok ( i , k)  ] ,
				[ ms_main ( i ) ] ) ;
		} ;	
		foreach ( j : speed_indices_list ) {
			add_rule ( [ ms_border_right ( j ) , ms_test_left ( i ) ] , 
				[ ms_border_right ( j ) , ms_test_left_fail ( i ) ] ) ;
			add_rule ( [ ms_main ( i ) , ms_test_left ( i ) ] , 
				[ ms_main ( j ) , ms_test_left_fail ( i ) ] ) ;
			add_rule ( [ ms_shrink_top ( RIGHT , i ) , ms_test_left ( i ) ] , 
				[ ms_shrink_top ( RIGHT , j ) , ms_test_left_fail ( i ) ] ) ;
		} ;
		add_rule ( [ ms_test_left_fail ( i ) , ms_main ( i ) ] ,   
			[ ms_main_test_fail_l ( i ) ] ) ;
		add_rule ( [ ms_main ( i ) , ms_test_right_fail ( i ) ] ,  
			[ ms_main_test_fail_r ( i ) ] ) ;
		add_rule ( [ ms_test_left_fail ( i ) , ms_main_test_fail_r ( i ) ] ,  
			[ ms_main ( i ) ] ) ;
		add_rule ( [ ms_main_test_fail_l ( i ) , ms_test_right_fail ( i ) ] ,  
			[ ms_main ( i ) ] ) ; 
		add_rule ( [ ms_main_test_fail_r ( i ) , ms_test_right_fail ( i ) ] ,  
			[ ms_main ( i ) ] ) ;
		add_rule ( [ ms_test_main_ok ( i ) , ms_test_right_fail ( i ) ] , 
			[ ms_main ( i ) ] ) ;
		add_rule ( [ ms_test_left_fail ( i ) , ms_main ( i ) , ms_test_right_fail ( i )  ] , 
			[ ms_main ( i ) ] ) ;
		add_rule ( [ ms_test_left_ok ( i ) , ms_main ( i ) , ms_test_right_fail ( i )  ] , 
			[ ms_main ( i ) ] ) ;
		//
		// FAIL on RIGHT
		//
		foreach ( k : 0 ... i -1 ) {
			foreach ( j : 0 ... k -1 ) {
				add_rule ( [ ms_test_right_wait ( i , k , j ) , ms_test_right_up ( i ) ] , 
					[ ms_test_right_fail ( i ) ] ) ;
			} ;
			add_rule ( [ ms_test_right_up ( i ) , ms_test_right_fail ( i ) ] , 
				[ ms_test_right_fail ( i ) ] ) ;
			add_rule ( [ ms_main_test_fail_l ( i ) , ms_test_right_ok ( i , k ) ] ,  
				[ ms_main ( i ) ] ) ;
			add_rule ( [ ms_test_left_fail ( i ) , ms_main ( i ) , ms_test_right_ok ( i , k )  ] ,
				[ ms_main ( i ) ] ) ;
		} ;	
		foreach ( j : speed_indices_list ) {
			foreach ( k : speed_indices_list ) {
				// Shrink on the right
				if ( ( k ≤ i ) && ( k ≤ j ) ) {
					foreach ( l : k ... i - 1  ) {
						add_rule ( [ ms_test_right ( i , l , k ) , ms_border_left ( j ) ] ,  
							[ ms_test_right_fail ( i ) , ms_border_left ( j ) ] ) ;
						add_rule ( [ ms_test_right ( i , l , k ) , ms_shrink_top ( LEFT , j ) ] ,  
							[ ms_test_right_fail ( i ) , ms_shrink_top ( LEFT , j ) ] ) ;
					} ;
				} ;
			} ;
		} ;			
		//
		// FAIL management
		//
		add_rule ( [ ms_test_left_ok ( i ) , ms_main_test_fail_r ( i ) ] , 
			[ ms_main ( i ) ] ) ;
	} ;
	

	//---------------------------------
	//  CHECKING
	//---------------------------------
	
	foreach ( i : speed_indices_list ) {
		foreach ( os : OS ) {
			foreach ( j : 0 ... i - 1 ) {
				// next 2 rules: case the second is the last -> no splitting
				add_rule ( [ ms_check_maybe ( i , j ) , ms_main_set ( j , os ) ] , 
					[ ms_main_set ( j , os ) ] ) ;
				add_rule ( [ ms_check_up ( i , j ) , ms_main_set ( j , os ) ] , 
					[ ms_check_ok ( i , j ) , ms_main_set ( j , os ) ] ) ;
				foreach ( k : j+1 ... i-1 ) {
					.ms_array := [] ;
					foreach ( l : j ... k - 1 ) {
						ms_array := ms_array _ ms_check_intersect ( i , k , l ) ;
					}
					add_rule ( [ ms_check_up ( i , j ) , ms_main_set ( k , os  ) ] , 
						ms_array _ ms_main_set ( k , os ) ) ;
				} ;
				foreach ( k : j + 1 ... i - 1 ) {  
						add_rule ( [ ms_check_maybe ( i , j ) , ms_main_set ( k , os ) ] , 
							[ ms_check ( i , j ) , ms_main_set ( k , os ) ] ) ;
				} ;	
				foreach ( k : j + 1 ... i - 1 ) {  
					add_rule ( [ ms_check_intersect ( i , k, j ) , ms_check_fail ( i , j ) ] , 
						[] ) ;
					foreach ( l : j + 1 ... k - 1 ) {  
						add_rule ( [ ms_check_intersect ( i , k, l ) , ms_check_fail ( i , j ) ] , 
							[ ms_check_fail ( i , j ) ] ) ;
					} ;
				} ;
			} ;
		} ;
		foreach ( j : 0 ... i - 1 ) {  
			add_rule ( [ ms_check_ok ( i , j ) , ms_id ( i ) ] , 
				[ ms_cross_back_ok ( i ) , ms_check_ok ( i , j ) ] ) ;
			add_rule ( [ ms_border_left ( i ) , ms_check_ok ( i , j ) ] , 
				[ ms_collect ( i , j ) , ms_border_left ( i ) ] ) ;
			foreach ( k : j + 1 ... i - 1 ) {  
				// CHECK is OK and finished
				add_rule ( [ ms_check ( i , j ) , ms_check_intersect ( i , k , j ) , ms_main ( j ) ] , 
					[ ms_check_ok ( i, j ) , ms_main ( j ) ] ) ;
				// CHECK is FAIL
				add_rule ( [ ms_check ( i , j ) , ms_main ( j ) ] , 
					[ ms_check_fail ( i , j ) ,  ms_main ( j ) ] ) ;
				add_rule ( [ ms_check ( i , j ) , ms_check_intersect ( i , k , j ) ] , 
					[] ) ;	// There should have been a main there
				// CHECK is FAIL
				add_rule ( [ ms_check ( i , j ) , ms_main ( k ) ] , 
					[ ms_check_fail ( i , j ) , ms_main ( j ) ] ) ;
				foreach ( l : j + 1 ... k - 1 ) {  
					 // CHECK is OK
					add_rule ( [ ms_check ( i , j ) , ms_check_intersect ( i , k , l ) ] , 
						[ ms_check ( i , j ) ] ) ;
					add_rule ( [ ms_check ( i , j ) , ms_check_intersect ( i , k , l ) , ms_main ( l ) ] , 
						[ ms_check ( i , j ) ,  ms_main ( l )] ) ;
					foreach ( m : j + 1 ... k - 1 ) {  
						if ( m != l ) {
							add_rule ( [ ms_check ( i , j ) , ms_check_intersect ( i , k , l ) , ms_main ( m ) ] , 
								[ ms_check_fail ( i , j ) , ms_main ( m ) ] ) ;
						} ;
					} ;
				} ;
			} ;
			foreach ( os : OS ) {
				foreach ( k : j + 1 ... i - 1 ) {  
					add_rule ( [ ms_check ( i , j ) , ms_check_intersect ( i , k , j ) , ms_main_set ( j , os ) ] , 
						[ ms_main_set ( j , os ) , ms_check_ok ( i , j ) ] ) ;
				} ;
			} ;
		} ;
	};
	
	
	//---------------------------------------
	//   DISPOSAL AND APPLACTION OF ID'S
	//---------------------------------------
				
	foreach ( i : speed_indices_list ) {
		foreach ( j : 0 ... i - 1 ) {  
			add_rule ( [ ms_collect ( i , j ) , ms_border_right ( j ) ] , 
				[ ms_ready ( i , [] )  ] ) ;
			foreach ( k : j ... i - 1 ) {  
				add_rule ( [ ms_collect ( i , j ) , ms_border_left ( k ) ] ,  
					[ ms_collect ( i , j ) ] ) ;
				add_rule ( [ ms_collect ( i , j ) , ms_id ( k ) ] ,  
					[ ms_cross_ok ( k ) , ms_collect ( i , j ) ] ) ;
				if ( j != k ) {
					add_rule ( [ ms_collect ( i , j ) , ms_border_right ( k ) ] , 
						[ ms_collect ( i , j ) ] ) ;
				} ;
				add_rule ( [ ms_collect ( i , j ) , ms_rule_bound ( k ) ] , 
					[ ms_collect ( i , j ) ] ) ;
				add_rule ( [ ms_collect ( i , j ) , ms_rule_middle ( k ) ] , 
					[ ms_collect ( i , j ) ] ) ;
				foreach ( l : speed_indices_list ) {
					add_rule ( [ ms_collect ( i , j ) , ms_if ( k , l ) ] , 
						[ ms_collect ( i , j ) ] ) ;
					add_rule ( [ ms_collect ( i , j ) , ms_then ( k , l ) ] , 
						[ ms_collect ( i , j ) ] ) ;
				} ;
			};
		} ;
		add_rule ( [ ms_cross_back_ok ( i ) , ms_rule_middle ( i ) ] ,  
			[ ms_cross_back ( i ) , ms_rule_middle ( i ) ] ) ;
		add_rule ( [ ms_cross_back ( i ) , ms_if ( i , i ) ] ,  
			[ ms_cross_back_ok ( i ) , ms_if_ok ( i , i ) ] ) ;
		add_rule ( [ ms_cross_back ( i ) , ms_rule_bound ( i  ) ] , 
			[ ms_cross_back_ok ( i ) , ms_rule_bound_fail ( i ) ] ) ; 
		add_rule ( [ ms_cross_back ( i ) , ms_rule_bound_fail ( i  ) ] , 
			[ ms_cross_back_ok ( i ) , ms_rule_bound_fail ( i ) ] ) ; 
		add_rule ( [ ms_cross_back_ok ( i ) , ms_border_right ( i ) ] , 
			[ ms_border_right ( i ) ] ) ;
		foreach ( k : speed_indices_list ) {
			add_rule ( [ ms_rule_bound ( i ) , ms_cross_ok ( k ) ] , 
				[ ms_cross ( k ) , ms_rule_bound ( i ) ] ) ;
			add_rule ( [ ms_if ( i , k ) , ms_cross ( k ) ] , 
				[ ms_cross_ok ( k ) , ms_if_ok ( i , k ) ] ) ;
			add_rule ( [ ms_rule_middle ( i ) , ms_cross ( k ) ] , 
				[ ms_cross ( k ) , ms_rule_middle_fail ( i ) ] ) ;
			add_rule ( [ ms_rule_middle_fail ( i ) , ms_cross ( k ) ] , 
				[ ms_cross ( k ) , ms_rule_middle_fail ( i ) ] ) ;
			foreach ( os : OS ) {
				add_rule ( [ ms_cross ( k ) , ms_main_set ( i , os ) ] ,
					[ ms_main_set ( i , os ) ] ) ;
			} ;
		} ;
	} ;

	
	//---------------------------------------
	//   RULE SELECTION
	//---------------------------------------
	
	foreach ( i : speed_indices_list ) {
		add_rule ( [ ms_border_right ( i ) , ms_ready ( i , [] ) ] , 
			[ ms_ready ( i , [] ) , ms_border_right ( i ) , ms_ready_up_1 () ] ) ; 
		foreach ( os : OS ) {
			add_rule ( [ ms_rule_bound_fail ( i ) , ms_ready ( i , os ) ] , 
				[ ms_ready_no ( i , os ) , ms_rule_bound_fail ( i ) ] ) ;
			add_rule ( [ ms_rule_middle_fail ( i ) , ms_ready ( i , os ) ] , 
				[ ms_ready_no ( i , os ) , ms_rule_middle_fail ( i )  ] ) ;
			add_rule ( [ ms_rule_bound ( i ) , ms_ready_no ( i , os ) ] , 
				[ ms_ready ( i , os ) , ms_rule_bound ( i ) ] ) ;
			// set could be os , it is a clearing strategy
			foreach ( l : speed_indices_list ) {
				add_rule ( [ ms_if ( i , l ) , ms_ready ( i , os ) ] , 
					[ ms_ready_no ( i , os ) , ms_if ( i , l ) ] ) ;
				add_rule ( [ ms_then ( i , l ) , ms_ready ( i , os ) ] , 
					[ ms_ready ( i , list_sorted_unique_insert ( os , l ) ) , ms_then ( i , l ) , ms_id_copy ( i , l ) ] ) ;
			} ;
		} ;
		foreach ( os : OS ) {
			foreach ( os2 : OS ) {
				add_rule ( [ ms_main_set ( i , os2 ) , ms_ready ( i , os ) ] , 
					[ ms_main_set ( i , os ) ] ) ;
			} ;
			add_rule ( [ ms_main_set ( i , os ) , ms_ready_up_1 ()] , 
				[ ms_main_set ( i , os ) , ms_ready_up_2 () ] ) ;
			foreach ( l : speed_indices_list ) {
				add_rule ( [ ms_id_copy ( i , l ) , ms_ready_up_2 () ] ,  
					[ ms_ready_up_2 (), ms_id_selected ( i , l )  ] ) ; 
			} ;
		} ;
		add_rule ( [ ms_border_left ( i ) , ms_ready_up_2 () ] , 
			[ ms_border_left ( i ) ] ) ;
	} ;


	//---------------------------------------
	//   SETTING OUTPUT
	//---------------------------------------

	foreach ( os : OS ) {
		.out := [ ms_fast_left ( os ) , ms_fast_right ( os ) ] ;
		foreach ( l : os ) {
			out := out _ ms_main ( l ) ;
		} ;

		foreach ( i : speed_indices_list ) {
		    slowers := [];
			foreach ( j : 0 ... i - 1 ) {  
				slowers := slowers _ j;
			} ;
            recursive_call_on_every_subset ( 
                slowers,
                (set) -> {
                    if (0 < #set) {
                        input := [ ms_main_set ( i , os ) ];
                        foreach (s : set) {
                            input := input _ ms_main ( s );
                        };
                        add_rule ( input , out ) ;
                    } ;
                }
            );

		} ;
	} ;
	
	foreach ( i : speed_indices_list ) {
		foreach ( os : OS ) {
			add_rule ( [ ms_fast_left ( os ) , ms_border_left ( i ) ] ,   
				ms_border_left_array ( os ) ) ;
			foreach ( l : speed_indices_list ) {
				add_rule ( [ ms_id_selected ( i , l )  , ms_fast_left ( os ) ] ,   
					[ ms_id ( l ) , ms_fast_left ( os ) ] ) ;
			} ;
			add_rule ( [ ms_fast_right ( os ) , ms_rule_bound ( i ) ] , 
				ms_rule_bound_array ( os ) _ ms_fast_right ( os ) ) ;
			add_rule ( [ ms_fast_right ( os ) , ms_rule_bound_fail ( i ) ] , 
				ms_rule_bound_array ( os ) _ ms_fast_right ( os ) ) ;
			add_rule ( [  ms_fast_right ( os ) , ms_rule_middle ( i ) ] , 
				ms_rule_middle_array ( os ) _ ms_fast_right ( os ) ) ;
			add_rule ( [ ms_fast_right ( os ) , ms_rule_middle_fail ( i ) ] , 
				ms_rule_middle_array ( os ) _ ms_fast_right ( os ) ) ;
			foreach ( l : speed_indices_list ) {
				add_rule ( [ ms_fast_right ( os ) , ms_if ( i , l ) ] ,  
					ms_if_array ( os , l ) _ ms_fast_right ( os ) ) ;
				add_rule ( [ ms_fast_right ( os ) , ms_if_ok ( i , l ) ] ,  
					ms_if_array ( os , l ) _ ms_fast_right ( os ) ) ;
				add_rule ( [ ms_fast_right ( os ) , ms_then ( i , l ) ] ,  
					ms_then_array ( os , l ) _ ms_fast_right ( os ) ) ;
			} ;
			add_rule ( [ ms_fast_right ( os ) , ms_border_right ( i ) ] ,   
				ms_border_right_array ( os ) ) ;
		} ;
	} ;


	} ;
} ;

