//
//  Library for the universal signal machine simulator
//
// This file provide the function to generate a simulating configuration from any initial configuration.
// The universal signal machine for the appropriate set of speeds of generated.
//

load "./lib_agc_iuniv.agc" ;


/** This function returns a simulating configuration for the corresponding universal signal machine.
- the simulated signal machine is recovered from the the configuration
- its speeds are collected
- in generates the universal signal machine for this set of speeds
- an empty configuration is generated 
- initial signals with all the encoding are set in
*/


.cretate_simulating_configuration .= (
	configuration_to_simulate
) -> {

	// GET the MACHINE

	.machine .= configuration_to_simulate . machine ;
	
	// COLLECT THE SPEEDS

	.speed_list := [] ; // . to create it in this context, i.e. locally
	foreach ( m_s : machine . get_meta_signal_list () ) {
		speed_list := list_sorted_unique_insert ( speed_list , m_s.speed ) ;
	};
	.speed_list .= .speed_list ; // Turn into a constant
	
	// CREATE THE SIMULATING MACHINE

	.simulating_signal_machine .= create_universal_simulator_speed ( speed_list ) ;
	
	// FIND THE MINIMUM DISTANCE BETWEEN SIGNALS (or at least a lower bound)

	. position_list := [] ;
	foreach ( sig : configuration_to_simulate . get_signal_list ) {
		position_list := list_sorted_unique_insert ( position_list , sig . birth_position ) ;
	} ;
	.position_previous := void ;
	.min_dist := 100 ; // random initial width 
	foreach ( position : position_list ) {
		if ( void != position_previous ) {
			min_dist := min ( min_dist , position - position_previous ) ;
		} ;
		position_previous := position ;
	} ;
	.half_width .= min_dist / 3 ;

	// ASSOCIATE NUMBERS TO META-SIGNALS
	// put them in list of list (first is speed)

	liste_speed_ms := [] ;
	foreach ( s : speed_list )  {
		liste_speed_ms := liste_speed_ms _ [ [] ] ;
	} ;	
	foreach ( m_s : machine . get_meta_signal_list () ) {
		i := list_get_index ( speed_list , m_s.speed ) ;
		liste_speed_ms [ i ] := liste_speed_ms [ i ] _ m_s ; 
	};

	// FUNCTION TO GET MS NUMBER

	.get_ms_number .= ( m_s ) -> {
		:= list_get_index ( liste_speed_ms [ list_get_index ( speed_list , m_s.speed ) ] , m_s ) + 1 ;
		// +1 to distinguish no signal from first signal
	} ;
		
	// GENERATE THE RULE ENCODING IN A LONG LIST

	.RULE_BOUND := "RULE_BOUND" ;
	.RULE_MIDDLE := "RULE_MIDDLE" ;
	.RULE_IF := "RULE_IF" ;
	.RULE_THEN := "RULE_THEN" ;

	.rule_encoding_sequence := [] ;
	foreach ( rule : machine . get_rule_list () ) {
		rule_encoding_sequence := rule_encoding_sequence _ [ RULE_BOUND ] ;
		foreach ( m_s : rule.out ) {
			j := list_get_index ( speed_list , m_s.speed ) ;
			n := get_ms_number ( m_s ) ;
			k := 0 ;
			while ( k < n ) {
				rule_encoding_sequence := rule_encoding_sequence _ [ [RULE_THEN , j ] ] ;
				k := k + 1 ;
			} ;
		} ;
		rule_encoding_sequence := rule_encoding_sequence _ [ RULE_MIDDLE ] ;
		foreach ( m_s : rule.in ) {
			j := list_get_index ( speed_list , m_s.speed ) ;
			n := get_ms_number ( m_s ) ;
			k := 0 ;
			while ( k < n ) {
				rule_encoding_sequence := rule_encoding_sequence _ [ [RULE_IF , j ] ] ;
				k := k + 1 ;
			} ;
		} ;
		rule_encoding_sequence := rule_encoding_sequence _ [ RULE_BOUND ] ;
	} ;
	rule_encoding_sequence .= rule_encoding_sequence ;
	.rule_encoding_length .= # rule_encoding_sequence ;
	
	
	// GENERATE THE SIMULATING CONFIGURATION

	.simulating_configuration .= simulating_signal_machine.create_configuration ();
	foreach ( sig : configuration_to_simulate . get_signal_list ) {
		.pos .= sig.birth_position ;
		.meta_signal .= sig.meta_signal ;
		.i .= list_get_index ( speed_list , meta_signal.speed ) ;
		// Compute left and right widths
		.width_left .= ( simulating_signal_machine.SPEED_RAPID + simulating_signal_machine.SPEED_MAX + meta_signal.speed ) / ( 2 * simulating_signal_machine.SPEED_RAPID ) * half_width ;
		.width_right .= ( simulating_signal_machine.SPEED_RAPID - simulating_signal_machine.SPEED_MAX - meta_signal.speed ) / ( 2 * simulating_signal_machine.SPEED_RAPID ) * half_width ;
		// ADD STRUCTURE
		ms_border_left ( i ) simulating_configuration.@ ( pos - width_left );	
		ms_main ( i ) simulating_configuration.@ pos ;	
		ms_border_right ( i ) simulating_configuration.@ pos + width_right ;	
		// ADD ID
		.nb .= get_ms_number ( meta_signal ) ;
		.k := 1 ;
		while ( k <= nb ) {
			ms_id ( i ) simulating_configuration.@ pos - width_left * k / ( nb + 1 ) ;	
			k := k + 1 ;
		} ;
		// ADD RULES
		.k := 1 ;
		.get_encoding_meta_signal := ( type ) -> {
			if ( RULE_BOUND == type ) {
				:= ms_rule_bound ( i , YES ) ;
			} else if ( RULE_MIDDLE == type ){
				:= ms_rule_middle ( i , OK ) ;
			};
			if ( RULE_THEN == type [ 0 ] ) {
				:= ms_then ( i , type [ 1 ] ) ;
			} ;
			if ( RULE_IF == type [ 0 ] ) {
				:= ms_if ( i , type [ 1 ] ) ;
			} ;
		} ;
		foreach ( type : rule_encoding_sequence ) {
			get_encoding_meta_signal ( type ) simulating_configuration.@ pos + width_right * k / ( rule_encoding_length + 1 ) ;
			k := k + 1 ;	
		}
	} ;
	
	:= simulating_configuration ;   // return
} ;


